<?php

namespace App\Http\Controllers;

use App\Http\Requests\SearchSurveyRequest;
use App\Survey;
use Carbon\Carbon;
use Illuminate\Http\Request;

class SurveyController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
    }

    /**
     * list Surveys
     * @return void
     */
    public function index()
    {

        $surveys = Survey::latest("created_at")->paginate(5);

        return view('surveys.list-surveys', compact('surveys'));
    }

    /**
     * @param SearchSurveyRequest $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function searchSurvey(SearchSurveyRequest $request)
    {

        $user = \Auth::user();
        if (!($user->hasRole("Admin") || $user->hasRole("Webmaster") || $user->hasRole("JournalisteWeb"))) {
            abort(403, 'Access denied');
        }

        $validated = $request->validated();
        $title = $validated["search_survey_title"];
        //display result
        $surveys = Survey::where('name', 'LIKE', "%{$title}%")
            ->latest("created_at")
            ->paginate(5);
        $surveys->appends(['search_survey_title' => $title]);
        return view('surveys.list-surveys', compact('surveys'), compact('title'));
    }

    /**
     * desactivate survey
     * @param Request $request
     * @return json
     */
    public function desactivate(Request $request)
    {
        $survey = Survey::find($request->id);
        $survey->status = 0;
        $survey->save();
        return response()->json(['success' => "Le sondage " . $survey->name . " a été désactivé avec succès"]);
    }

    /**
     * activate survey
     * @param Request $request
     * @return json
     */
    public function activate(Request $request)
    {
        $survey = Survey::find($request->id);
        $survey->status = 1;
        $survey->save();
        return response()->json(['success' => "Le sondage " . $survey->name . " a été activé avec succès"]);
    }

    /**
     * delete a survey
     *
     * @param int $surveyId
     * @return void
     */
    public function deleteSurvey($surveyId)
    {
        $survey = Survey::find($surveyId);
        $user = \Auth::user();
        if (!($user->hasRole("Admin") || $user->hasRole("Webmaster") || $user->hasRole("JournalisteWeb"))) {
            abort(403, 'Access denied');
        }
        $survey->answersSurveys()->delete();
        $survey->delete();
        return response()->json(['redirect' => route('surveysList')]);
    }

    /**
     * new survey form
     *
     * @return void
     */
    public function newSurvey()
    {
        return view('surveys.create-survey');
    }

    public function ajxNewSurvey(Request $request)
    {
        $survey = new Survey();
        $startDate = Carbon::createFromFormat('d-m-Y', $request->startDate)->format('Y-m-d');
        $endDate = Carbon::createFromFormat('d-m-Y', $request->endDate)->format('Y-m-d');
        $survey->name = $request->name;
        $survey->startDate = $startDate;
        $survey->endDate = $endDate;
        $survey->save();
        foreach ($request->answers as $answer) {
            $survey->answersSurveys()->create(['name' => $answer]);
        }
        return response()->json(['redirect' => route('surveysList')]);
    }

    /**
     * edit survey form
     *
     * @param int $surveyId
     * @return void
     */
    public function editSurvey($surveyId)
    {
        $survey = Survey::where('id', '=', $surveyId)->with('answersSurveys')->first();
        return view('surveys.edit-survey')
            ->with(compact('survey'));
    }

    public function ajxUpdateSurvey(Request $request)
    {
        $survey = Survey::findOrFail($request->surveyId);
        $survey->answersSurveys()->delete();
        $survey->delete();
        $startDate = Carbon::createFromFormat('d-m-Y', $request->startDate)->format('Y-m-d');
        $endDate = Carbon::createFromFormat('d-m-Y', $request->endDate)->format('Y-m-d');
        $survey->name = $request->name;
        $survey->startDate = $startDate;
        $survey->endDate = $endDate;
        $survey->save();
        foreach ($request->answers as $answer) {
            $survey->answersSurveys()->create(['name' => $answer]);
        }
        return response()->json(['redirect' => route('surveysList')]);
    }
}
