<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Requests\CreateMarchiCategoryRequest;
use App\Http\Requests\UpdateMarchiCategoryRequest;
use App\Http\Requests\CreateMarchiProductRequest;
use App\Http\Requests\UpdateMarchiProductRequest;
use App\Http\Requests\CreateMarchiDayRequest;
use App\Http\Requests\UpdateMarchiDayRequest;

use App\MarchiCategory;
use App\MarchiProduct;
use App\MarchiPrice;

use Carbon\Carbon;

class MarchiController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        
    }

    /**
     * list of categories
     * 
     * @return void
     */
    public function CategoryList(){
        $categories = MarchiCategory::orderBy("order",'asc')->get();
        return view('marchi.list-category')->with(compact("categories"));
    }

    /**
     * new category form
     * 
     * @return void
     */
    public function newCategory(){
        return view('marchi.create-category');
    }

    /**
     * add new marchi category request
     * 
     * @param CreateMarchiCategoryRequest $request
     * 
     * @return void
     */
    public function ajxNewCategory(CreateMarchiCategoryRequest $request){
        $cntCat=MarchiCategory::count();
        $cntCat++;
        $cat=new MarchiCategory();
        $cat->title=$request->get("title");
        $cat->img=$request->get("img");
        $cat->img_alt=($request->get("img_alt")) ? $request->get("img_alt") : $request->get("title");
        $cat->summary=$request->get("summary");
        $cat->meta_title=$request->get("meta_title");
        $cat->meta_keywords=$request->get("meta_keywords");
        $cat->meta_description=$request->get("meta_description");
        $cat->order=$cntCat;

        $alias=$request->get("title");
        $alias=str_replace(" ",'-',$alias);
        $alias=str_replace("/",'-',$alias);
        $alias=urlencode($alias);
        $cat->url_rewrite=$alias;
        $cat->save();

        return response()->json(['redirect'=>route('marchiCategoriesList')]);
    }

    /**
     * edit category
     * 
     * @param \App\MarchiCategory $category
     * @return \Illuminate\Http\Response
     */
    public function editCategory(MarchiCategory $category){
        $cntCat=MarchiCategory::count();
        return view('marchi.edit-category')->with(compact("category"))->with(compact("cntCat"));
    }

    /**
     * update category ajax request
     * 
     * @param UpdateMarchiCategoryRequest $request
     * @param MarchiCategory $category
     * 
     * @return void
     */
    public function ajxUpdateCategory(UpdateMarchiCategoryRequest $request,MarchiCategory $category){

        $oldOrder=$category->order;

        if($oldOrder>$request->get("order")){
            MarchiCategory::where("order",">=",$request->get("order"))->where("order","<",$oldOrder)->increment("order");
        }
        if($oldOrder<$request->get("order")){
            MarchiCategory::where("order",">",$request->get("order"))->where("order","<=",$oldOrder)->decrement("order");
        }

        $category->title=$request->get("title");
        $category->img=$request->get("img");
        $category->img_alt=($request->get("img_alt")) ? $request->get("img_alt") : $request->get("title");
        $category->summary=$request->get("summary");
        $category->meta_title=$request->get("meta_title");
        $category->meta_keywords=$request->get("meta_keywords");
        $category->meta_description=$request->get("meta_description");
        $category->order=$request->get("order");
        $category->save();

        return response()->json(['redirect'=>route('marchiCategoriesList')]);

    }

    /**
     * delete a category
     * 
     * @param MarchiCategory $category
     * @return void
     */
    public function deleteCategory($category){
        $order=$category->order;
        $category->delete();
        $categories=MarchiCategory::where("order",'>',$order)->orderBy("order",'asc')->get();
        foreach($categories as $one_category){
            $one_category->order=$one_category->order-1;
            $one_category->save();
        }

        return response()->json(['redirect'=>route('marchiCategoriesList')]);
    }

    /**
     * list of products
     * 
     * @return void
     */
    public function ProductList(){
        $products=MarchiProduct::orderBy("order",'asc')->with("marchiCategory")->get();       
        return view('marchi.list-product')->with(compact("products"));
    }

    /**
     * new product form
     * 
     * @return void
     */
    public function newProduct(){
        $categories=MarchiCategory::get();
        return view('marchi.create-product')->with(compact("categories"));
    }

    /**
     * add new marchi product request
     * 
     * @param CreateMarchiProductRequest $request
     * 
     * @return void
     */
    public function ajxNewProduct(CreateMarchiProductRequest $request){
        $cntProd=MarchiProduct::count();
        $cntProd++;
        $product=new MarchiProduct();
        $product->title=$request->get("title");
        $product->marchi_category_id=$request->get("marchi_category_id");
        $product->img=$request->get("img");
        $product->img_alt=($request->get("img_alt")) ? $request->get("img_alt") : $request->get("title");
        $product->summary=$request->get("summary");
        $product->meta_title=$request->get("meta_title");
        $product->meta_keywords=$request->get("meta_keywords");
        $product->meta_description=$request->get("meta_description");
        $product->order=$cntProd;

        $alias=$request->get("title");
        $alias=str_replace(" ",'-',$alias);
        $alias=str_replace("/",'-',$alias);
        $alias=urlencode($alias);
        $product->url_rewrite=$alias;
        $product->save();

        return response()->json(['redirect'=>route('marchiProductsList')]);
    }

    /**
     * edit product
     * 
     * @param \App\MarchiProduct $product
     * @return \Illuminate\Http\Response
     */
    public function editProduct(MarchiProduct $product){
        $categories=MarchiCategory::get();
        $cntProd=MarchiProduct::count();
        return view('marchi.edit-product')->with(compact("product"))->with(compact("cntProd"))->with(compact("categories"));
    }

    /**
     * update product ajax request
     * 
     * @param UpdateMarchiProductRequest $request
     * @param MarchiProduct $product
     * 
     * @return void
     */
    public function ajxUpdateProduct(UpdateMarchiProductRequest $request,MarchiProduct $product){

        $oldOrder=$product->order;

        if($oldOrder>$request->get("order")){
            MarchiProduct::where("order",">=",$request->get("order"))->where("order","<",$oldOrder)->increment("order");
        }
        if($oldOrder<$request->get("order")){
            MarchiProduct::where("order",">",$request->get("order"))->where("order","<=",$oldOrder)->decrement("order");
        }

        $product->title=$request->get("title");
        $product->img=$request->get("img");
        $product->marchi_category_id=$request->get("marchi_category_id");
        $product->img_alt=($request->get("img_alt")) ? $request->get("img_alt") : $request->get("title");
        $product->summary=$request->get("summary");
        $product->meta_title=$request->get("meta_title");
        $product->meta_keywords=$request->get("meta_keywords");
        $product->meta_description=$request->get("meta_description");
        $product->order=$request->get("order");
        $product->save();

        return response()->json(['redirect'=>route('marchiProductsList')]);

    }

    /**
     * delete a product
     * 
     * @param MarchiProduct $product
     * @return void
     */
    public function deleteProduct(MarchiProduct $product){
        $order=$product->order;
        $product->delete();
        $products=MarchiProduct::where("order",'>',$order)->orderBy("order",'asc')->get();
        foreach($products as $one_product){
            $one_product->order=$one_product->order-1;
            $one_product->save();
        }

        return response()->json(['redirect'=>route('marchiProductsList')]);
    }

    /**
     * days list
     * 
     * @return void
     */
    public function days(){
        $days = MarchiPrice::select("day_date")->latest("day_date")->groupBy('day_date')->Paginate(100);

        return view('marchi.list-days')->with(compact("days"));
    }

    /**
     * create new day form
     * 
     * @return void
     */
    public function newDay(){
        $categories=MarchiCategory::get();
        $date=Carbon::now();

        return view('marchi.create-day')->with(compact("categories"))->with(compact("date"));
    }

    /**
     * create day ajax request
     * 
     * @param CreateMarchiDayRequest $request
     * 
     * @return void
     */
    public function ajxNewDay(CreateMarchiDayRequest $request){

        $date=Carbon::createFromFormat("d/m/Y", $request->get("day_date"));

        MarchiPrice::whereDayDate($date->format("Y-m-d"))->delete();

        foreach($request->all() as $key=>$value){
            if(substr($key,0,18)=="product_min_price_"){
                $productId=substr($key,18,(strlen($key)-18));
                if($request->get("product_check_".$productId)){
                    $price=new MarchiPrice();
                    $price->marchi_product_id=$productId;
                    $price->day_date=$date;
                    $price->price_min=$value;
                    $price->price_max=$request->get("product_max_price_".$productId);
                    $price->save();
                }
            }
        }

        return response()->json(['redirect'=>route('marchiDays')]);

    }

    /**
     * edit day form
     * 
     * @param string $day
     * @return void
     */
    public function editDay($day){
        $date=Carbon::createFromFormat("Y-m-d", $day);
        $categories=MarchiCategory::get();
        return view('marchi.edit-day')->with(compact("categories"))->with(compact("date"));
    }

    /**
     * update day ajax request
     * 
     * @param UpdateMarchiDayRequest $request
     * @param string $day
     * 
     * @return void
     */
    public function ajxUpdateDay(UpdateMarchiDayRequest $request,$day){

        $date=Carbon::createFromFormat("Y-m-d", $day);

        MarchiPrice::whereDayDate($day)->delete();

        foreach($request->all() as $key=>$value){
            if(substr($key,0,18)=="product_min_price_"){
                $productId=substr($key,18,(strlen($key)-18));
                if($request->get("product_check_".$productId)){
                    $price=new MarchiPrice();
                    $price->marchi_product_id=$productId;
                    $price->day_date=$date;
                    $price->price_min=$value;
                    $price->price_max=$request->get("product_max_price_".$productId);
                    $price->save();
                }
                
            }
        }

        return response()->json(['redirect'=>route('marchiDays')]);

    }

    /**
     * delete day
     * 
     * @param string deleteDay
     * @return void
     */
    public function deleteDay($day){
        $date=Carbon::createFromFormat("Y-m-d", $day);

        MarchiPrice::whereDayDate($day)->delete();

        return response()->json(['redirect'=>route('marchiDays')]);
    }
    

}
